/* 

                          Firewall Builder

                 Copyright (C) 2003 NetCitadel, LLC

  Author:  Vadim Kurland     vadim@fwbuilder.org

  $Id: findDialog.cpp,v 1.13 2006/06/13 06:54:24 vkurland Exp $

  This program is free software which we release under the GNU General Public
  License. You may redistribute and/or modify this program under the terms
  of that license as published by the Free Software Foundation; either
  version 2 of the License, or (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
 
  To get a copy of the GNU General Public License, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

*/


#include "config.h"
#include "global.h"
#include "utils.h"

#include "findDialog.h"
#include "ObjectManipulator.h"
#include "FWWindow.h"
#include "FWBTree.h"
#include "FWBSettings.h"

#include "fwbuilder/FWObjectDatabase.h"
#include "fwbuilder/FWReference.h"
#include "fwbuilder/RuleSet.h"
#include "fwbuilder/RuleElement.h"
#include "fwbuilder/Firewall.h"
#include "fwbuilder/IPService.h"
#include "fwbuilder/ICMPService.h"
#include "fwbuilder/TCPService.h"
#include "fwbuilder/UDPService.h"

#include <qcombobox.h>
#include <qcheckbox.h>
#include <qapplication.h>
#include <qcursor.h>
#include <qregexp.h>
#include <qlineedit.h>
#include <qpushbutton.h>
#include <qmessagebox.h>
#include <qtimer.h>

#include <iostream>

using namespace libfwbuilder;
using namespace std;

#define MAX_SEARCH_ITEMS_COUNT 10

findDialog::findDialog(QWidget *p) : findDialog_q(p), treeSeeker()
{
    lastFound=NULL;
    lastTextSearch=""; 
    lastAttrSearch=""; 

    findText->setFocus();
}

void findDialog::setObject(FWObject *o)
{
    reset();
    findText->setCurrentText( QString::fromUtf8(o->getName().c_str()) );
}

void findDialog::reset()
{
    lastFound=NULL;
    lastTextSearch="";
    treeSeeker=mw->db()->tree_begin();
}

void findDialog::findTextChanged(const QString &ns)
{
    if (ns!=lastTextSearch)  reset();
    lastTextSearch=ns;
}

void findDialog::findAttrChanged(const QString &ns)
{
    if (ns!=lastAttrSearch)  reset();
    lastAttrSearch=ns;
}

void findDialog::find()
{
    if (findText->currentText().isEmpty() && findAttr->currentText().isEmpty()) return;

    if (findText->currentText() != findText->text(0))
    {
        if (findText->count()>=MAX_SEARCH_ITEMS_COUNT)
            findText->removeItem(MAX_SEARCH_ITEMS_COUNT-1);

        findText->insertItem( findText->currentText() , 0 );

        if (fwbdebug)
        {
            qDebug("findDialog::find() : findText->text(0)=%s",
                   findText->text(0).latin1());
        }
    }

    if (findAttr->currentText() != findAttr->text(0))
    {
        if (findAttr->count()>=MAX_SEARCH_ITEMS_COUNT)
            findAttr->removeItem(MAX_SEARCH_ITEMS_COUNT-1);

        findAttr->insertItem( findAttr->currentText() , 0 );

        if (fwbdebug)
            qDebug("findDialog::find() : findAttr->text(0)=%s",
                   findAttr->text(0).latin1());
    }

    findNext();
}

bool findDialog::matchName(const QString &name)
{
    QString s=findText->currentText();
    if (s.isEmpty()) return true;

    bool res=false;

    if (useRegexp->isChecked()) res= ( name.find( QRegExp(s) )!=-1 );
    else                        res= ( name == s );

    return res;
}

bool findDialog::matchAttr(libfwbuilder::FWObject *obj)
{
    QString s=findAttr->currentText();
    if (s.isEmpty()) return true;

    bool res=false;
    int  attrN = attribute->currentItem();

    switch (attrN) {
    case 0:   // Address
    {
        Address *a = Address::cast(obj);
        if (a!=NULL)
        {
            QString addr = a->getAddress().toString().c_str();
            if (useRegexp->isChecked()) res= ( addr.find( QRegExp(s) )!=-1 );
            else                        res= ( addr == s );
        }
        break;
    }
    case 1:   // port
        if (TCPService::cast(obj)!=NULL || UDPService::cast(obj)!=NULL)
        {
            if (useRegexp->isChecked()) 
            {
                QString port;
                port.setNum(obj->getInt("src_range_start"));
                res |= ( port.find( QRegExp(s) )!=-1 );
                port.setNum(obj->getInt("src_range_end"));
                res |= ( port.find( QRegExp(s) )!=-1 );
                port.setNum(obj->getInt("dst_range_start"));
                res |= ( port.find( QRegExp(s) )!=-1 );
                port.setNum(obj->getInt("dst_range_end"));
                res |= ( port.find( QRegExp(s) )!=-1 );
            } else
            {
                int port = s.toInt();
                res |= (port == obj->getInt("src_range_start"));
                res |= (port == obj->getInt("src_range_end"));
                res |= (port == obj->getInt("dst_range_start"));
                res |= (port == obj->getInt("dst_range_end"));
            }
            break;
        }
    case 2:   // protocol num.
        if (IPService::cast(obj)!=NULL)
        {
            if (useRegexp->isChecked()) 
            {
                QString proto;
                proto.setNum(obj->getInt("protocol_num"));
                res |= ( proto.find( QRegExp(s) )!=-1 );
            } else
            {
                int proto = s.toInt();
                res |= (proto == obj->getInt("protocol_num"));
            }
            break;
        }
    case 3:   // icmp type
        if (ICMPService::cast(obj)!=NULL)
        {
            if (useRegexp->isChecked()) 
            {
                QString icmptype;
                icmptype.setNum(obj->getInt("type"));
                res |= ( icmptype.find( QRegExp(s) )!=-1 );
            } else
            {
                int icmptype = s.toInt();
                res |= (icmptype == obj->getInt("type"));
            }
            break;
        }
    }

    return res;
}

void findDialog::findNext()
{
    if (findText->currentText().isEmpty() &&
        findAttr->currentText().isEmpty()) return;

    if (findText->count()>10)  findText->removeItem(0);
    if (findAttr->count()>10)  findAttr->removeItem(0);

    FWObject *o=NULL;

loop:
    QApplication::setOverrideCursor( QCursor( Qt::WaitCursor) );

    for (; treeSeeker!=mw->db()->tree_end(); ++treeSeeker)
    {
        o = *treeSeeker;

        if( RuleElement::cast(o->getParent())!=NULL)
        {
            if (! searchInRules->isChecked()) continue;
        } else
        {
/* if not in rules, then in the tree. */
            if (! searchInTree->isChecked()) continue;
        }

        if (FWReference::cast(o)!=NULL)
        {
            FWReference *r=FWReference::cast(o);
            if ( matchName( QString::fromUtf8(r->getPointer()->getName().c_str()) ) &&
                 matchAttr( r->getPointer() )) break;
        } else
        {
            if (matchName( QString::fromUtf8(o->getName().c_str())) &&
                matchAttr( o )) break;
        }
    }

    QApplication::restoreOverrideCursor();

    if (treeSeeker==mw->db()->tree_end())
    {
        reset();

        if ( QMessageBox::warning(
              this,"Firewall Builder", 
              tr("Search hit the end of the object tree."),
              tr("&Continue at top"), tr("&Stop"), QString::null, 0, 1 )==0 ) goto loop;

        return;
    }
    assert(o!=NULL);

/* found object. Shift iterator so it does not return the same object
 * when user hits 'find next'
 */
   
    ++treeSeeker;

    if (FWReference::cast(o)!=NULL && RuleElement::cast(o->getParent())!=NULL)
    {
        mw->ensureObjectVisibleInRules( FWReference::cast(o) );
        QTimer::singleShot(200, this, SLOT(makeActive()) );
        return;
    } 

    if (Group::cast(o->getParent())!=NULL && 
        !FWBTree::isSystem(o->getParent()))
    {
        om->openObject( o->getParent() );
        om->editObject( o->getParent() );
        QTimer::singleShot(200, this, SLOT(makeActive()) );
        return;
    }

    if (fwbdebug)
    {
        qDebug("Found object: o=%p  id=%s  name=%s  type=%s",
               o, o->getId().c_str(),o->getName().c_str(),o->getTypeName().c_str());
    }

    om->openObject( o );
    om->editObject( o );

    QTimer::singleShot(200, this, SLOT(makeActive()) );
}

void findDialog::makeActive()
{
    setActiveWindow();
}

void findDialog::showEvent( QShowEvent *ev)
{
    st->restoreGeometry(this, QRect(200,100,330,140) );
    QDialog::showEvent(ev);
    
    useRegexp->setChecked( st->getBool("Search/useRegexp") );
    searchInTree->setChecked(  st->getBool("Search/findInTree" ) );
    searchInRules->setChecked( st->getBool("Search/findInRules") );

    findText->setFocus();
}

void findDialog::hideEvent( QHideEvent *ev)
{
    st->saveGeometry(this);
    QDialog::hideEvent(ev);

    st->setBool("Search/useRegexp",    useRegexp->isChecked() );
    st->setBool("Search/findInTree",  searchInTree->isChecked() );
    st->setBool("Search/findInRules", searchInRules->isChecked() );
}
    
